' Hexapawn -- A game invented by Martin Gardner to demonstrate machine learning.
' Graphics are based on a VSauce2 YouTube video about same.
' Rev 1.0.0 William M Leue 27-June-2023

option default integer
option base 1
option angle degrees

' Constants
const NB_TWO    = 2
const NB_FOUR   = 11
const NB_SIX    = 11
const NUMB      = NB_TWO+NB_FOUR+NB_SIX
const UBNUM     = NUMB+1
const MAX_MOVES = 8

const NPLAYER = 2
const PWHITE  = 1
const PBLACK  = 2

const NBCOLS  = 4
const BBLUE   = 1
const BGREEN  = 2
const BVIOLET = 3
const BORANGE = 4

const NMOVES  = 3
const ADOWN   = 1   ' Computer move directions
const ALEFT   = 2
const ARIGHT  = 3
const USUP    = 4   ' User move directions
const USLEFT  = 5
const USRIGHT = 6
const SQINDEX = 1
const BDINDEX = 2
const DRINDEX = 3

' box parameters
const BWIDTH    = 60
const BHEIGHT   = 90
const BSEP      = 4
const BMARG     = 3
const BEAD_RAD  = 2
const BEAD_XOFF = 5
const BEAD_YOFF = 10
const MAGBOX_X  = 60
const MAGBOX_Y = 370
const MAG      = 2

' arrow params
const ANVERT   = 8
const ATHICK1  = 3
const ASHLEN1  = 8
const AHDLEN1  = 6
const AHDWID1  = 6
const ATHICK2  = 5
const ASHLEN2  = 9
const AHDLEN2  = 6
const AHDWID2  = 6

' board values
const BSIDE   = 3
const CSIZE   = 50
const PRAD    = int(0.35*CSIZE)
const BYPOS   = 370
const BDMARG  = 4

' keyboard commands
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const CR    = 10
const ESC   = 27

' user moves
const AHEAD = 1
const LEFT_CAPTURE = 2
const RIGHT_CAPTURE = 3

' Games summary
const GSX = 600
const GSW = 199
const GSH = 219
const MCHARS = 22
const MAX_GAMES = 2*MCHARS
const WGM = 20
const WGX = GSX+WGM
const WGW = GSW-2*WGM
const WGH = int(2*GSH/5)
const WGY = BYPOS+GSH-WGM-WGH

' Message parameters
const MSG_Y = 325

' Globals
dim bboards(NUMB+1, BSIDE, BSIDE)
dim bmoves(NUMB+1, NBCOLS, 3)
dim beads(NUMB+1, NBCOLS)
dim nbeads(NUMB)
dim bcolors(NBCOLS) = (rgb(blue), rgb(green), rgb(229, 102, 255), rgb(orange))
dim mcolors$(NBCOLS) = ("Blue", "Green", "Lavender", "Orange")
dim lraoff = 0
dim sx, sy
dim running = 0
dim board(BSIDE, BSIDE)
dim bx, by
dim cur_col = 0
dim selected_move = 0
dim prev_selected_move = 0
dim matching_box = 0
dim prev_matching_box = 0
dim apars1(4), apars2(4)
dim moves(NBCOLS, 3)
dim ply = 0
dim pnames$(2) = ("WHITE", "BLACK")
dim mover = 0
dim ngames = 0
dim games(MAX_GAMES)
dim cmoves(MAX_MOVES, 2)
dim cptr = 0
dim flip = 0

' Main Program
'open "debug.txt" for output as #1
SetArrowParams
ReadBoxContents
ShowHelp
cls
text mm.hres\2, 1, "Hexapawn!", "CT", 5, 2, rgb(green)
running = 1
InitGameSeries
do
  InitGame
  DrawBoxes 1
  PlayGame
  DrawBox UBNUM, 0, MAGBOX_X, MAGBOX_Y, 2, 0, 1
  WriteTacticalMessage ""
  pause 1000
  WriteMessage "New Game", -1
  pause 1000
loop until not running
end

' Create parameter arrays for arrows
sub SetArrowParams
  apars1(1) = ATHICK1
  apars1(2) = ASHLEN1
  apars1(3) = AHDLEN1
  apars1(4) = AHDWID1
  apars2(1) = ATHICK2
  apars2(2) = ASHLEN2
  apars2(3) = AHDLEN2
  apars2(4) = AHDWID2
end sub

' Read the box contents data into the bboards() array
sub ReadBoxContents
  local bx, row, col, bd
  for bx = 1 to NUMB
    for row = 1 to BSIDE
      for col = 1 to BSIDE
        read bboards(bx, row, col)
      next col
    next row
    for bd = 1 to NBCOLS
      read bmoves(bx, bd, SQINDEX)
      read bmoves(bx, bd, DRINDEX)
      if bmoves(bx, bd, SQINDEX) > 0 then
        bmoves(bx, bd, BDINDEX) = bd
      else
        bmoves(bx, bd, BDINDEX) = 0
      end if
    next bd
  next bx
end sub

' Initialize for a game series
sub InitGameSeries
  local i
  ngames = 0
  for i = 1 to MAX_GAMES
    games(i) = 0
  next i
  InitBeads
end sub

' Initialize for a new game
sub InitGame
  local col, i
  ply = 1
  for col = 1 to BSIDE
    board(2, col) = 0
  next col
  for col = 1 to BSIDE
    board(1, col) = PBLACK
    board(BSIDE, col) = PWHITE
  next col
  prev_selected_row = 0
  prev_selected_col = 0
  selected_col = 0
  selected_row = 0
  matching_box = 0
  prev_matching_box = 0
  cptr = 0
  flip = 0
  restore
  ReadBoxContents
  DrawBoard
  for i = 1 to MAX_MOVES
    cmoves(i,1) = 0
    cmoves(i,2) = 0
  next i
end sub

' Initialize the beads in each box
sub InitBeads
  local bx, bd, bc
  for bx = 1 to NUMB
    nbeads(bx) = 0
    for bd = 1 to NBCOLS
      beads(bx, bd) = 0
      bc = bmoves(bx, bd, BDINDEX)
      if bc > 0 then
        beads(bx, bd) = 1
        inc nbeads(bx)
      end if
    next bd
  next bx
end sub

' Draw the Board
sub DrawBoard
  local bw, bh, row, col, x1, y1, x2, y2, ec, fc, x, y
  bw = BSIDE*CSIZE + 2*BDMARG
  bh = BSIDE*CSIZE + 2*BDMARG
  bx = mm.hres\2 - bw\2
  by = BYPOS
  box bx, by, bw, bh,, rgb(white), rgb(white)
  box bx+BDMARG, by+BDMARG, bw-2*BDMARG, bh-2*BDMARG,, rgb(blue)
  x1 = bx+BDMARG : x2 = bx+bw-BDMARG
  for row = 1 to 2
    y1 = by+BDMARG + row*CSIZE
    line x1, y1, x2, y1,, rgb(blue)
  next row
  y1 = by+BDMARG : y2 = by+BH-BDMARG
  for col = 1 to 2
    x1 = bx+BDMARG + col*CSIZE
    line x1, y1, x1, y2,, rgb(blue)
  next col
  for row = 1 to BSIDE
    y = by + BDMARG + (row-1)*CSIZE + CSIZE\2
    for col = 1 to BSIDE
      x = bx + BDMARG + (col-1)*CSIZE + CSIZE\2
      select case board(row, col)
        case PBLACK
          ec = rgb(black)
          fc = rgb(blue)
        case PWHITE
          ec = rgb(black)
          fc = rgb(yellow)
        case else
          ec = rgb(white)
          fc = rgb(white)
      end select
      circle x, y, PRAD,,, ec, fc
    next col
  next row
end sub

' Play the HexaPawn game
' Note: if user 1st move is col = 3, then the 'flip' var is set to 1
' and remains at one for the duration of the game. This mirrors all
' the boxes L/R. (they get flipped back for the next game.)
sub PlayGame
  local winner = 0
  local nmoves, umove, bx, flip
  local moves(NBCOLS, 3)
  ply = 0
  do
    inc ply
    mover = PWHITE
    winner = IsWinner()
    if winner then exit do
    FindAllUserMoves moves(), nmoves
    DrawUserMoves moves(), nmoves
    if nmoves > 0 then
      WriteMessage "Your Move", -1
      prev_selected_move = 0
      selected_move = 1
      SelectUserMove moves(), nmoves, umove
      if (ply = 1) and (umove = 3) then
        flip = 1
        MirrorBoxes
      end if
      DoUserMove moves(), umove
    else
      WriteMessage "You Have No Moves", 1
      winner = PWHITE
      exit loop
    end if
    mover = PBLACK
    inc ply
    winner = IsWinner()
    if winner then exit do
    pause 1000
    WriteMessage "Computer's Move", -1
    bx = FindMatchingBox()
    if bx = 0 then
      WriteMessage "Error -- No Matching Box", -1
      end
    end if
    HiliteBox bx
    MakeUseBox bx
    DrawBox UBNUM, bx, MAGBOX_X, MAGBOX_Y, 2, ply, 0
    DoComputerMove UBNUM, bx
    pause 1000
  loop until winner
  if winner = PWHITE then
    WriteMessage "You Win!", 1
    TrainComputer
  else
    WriteMessage "Computer Wins!", 1
  end if
  inc ngames
  games(ngames) = winner
  DrawGamesSummary
  if flip then
    MirrorBoxes
    flip = 0
  end if
end sub

' Mirror all boxes L/R
' Have to flip both the pieces and the moves.
' This happens ONLY when the user opens with his piece on column 3,
' because there are only enough boxes for the 'left-hand' game.
sub MirrorBoxes
  local bx, row, col, t, sq, bd, dir, fsq, fcol, fdir
  for bx = 1 to NUMB
    for row = 1 to BSIDE
      t = bboards(bx, row, 1)
      bboards(bx, row, 1) = bboards(bx, row, BSIDE)
      bboards(bx, row, BSIDE) = t
    next row
    for bd = 1 to NBCOLS
      sq = bmoves(bx, bd, SQINDEX)
      dir = bmoves(bx, bd, DRINDEX)
      if sq > 0 then
        col = ((sq-1) mod BSIDE) +1
        row = ((sq-1)\BSIDE) + 1
        fcol = BSIDE-col+1
        fsq = (row-1)*BSIDE + fcol
        select case dir
          case ADOWN : fdir  = ADOWN
          case ALEFT : fdir  = ARIGHT
          case ARIGHT : fdir = ALEFT
        end select
        bmoves(bx, bd, SQINDEX) = fsq
        bmoves(bx, bd, DRINDEX) = fdir
      end if
    next bd
  next bx
  DrawBoxes 1
end sub

' Draw arrows that represent all possible user moves  from the current board
' position. There are never more than 4 such moves.
sub DrawUserMoves moves(), nmoves
  local row, col, dir, i, cx, cy, rx, ry, fs, ss
  local float angle
  fs = BWIDTH - 2*BMARG
  ss = fs\3
  for i = 1 to nmoves
    row = moves(i, 1) : col = moves(i, 2) : dir = moves(i, 3)
    cx = bx + (col-1)*CSIZE + CSIZE\2
    cy = by + (row-1)*CSIZE + CSIZE\2
    select case dir
      case USUP
        rx = cx+5
        ry = cy - ss
        angle = 90.0
      case USLEFT
        rx = cx + lraoff
        ry = cy - lraoff
        angle = 135.0
      case USRIGHT
        rx = cx + lraoff
        ry = cy - lraoff
        angle = 45.0
    end select
    DrawArrow apars1(), rx, ry, angle, i, 2
  next i
end sub

' Make a box to use for the next computer move. This box will be displayed in
' magnified form next to the board.
sub MakeUseBox bx
  local row, col, bd, sq, dir
  if bx = 0 then
    cls
    print "Error -- zero box index"
    end
  end if
  for row = 1 to BSIDE
    for col = 1 to BSIDE
      bboards(UBNUM, row, col) = bboards(bx, row, col)
    next col
  next row
  for bd = 1 to NBCOLS
    beads(UBNUM, bd) = beads(bx, bd)
    if beads(bx, bd) > 0 then
      sq = bmoves(bx, bd, SQINDEX)
      dir = bmoves(bx, bd, DRINDEX)
      bmoves(UBNUM, bd, SQINDEX) = sq
      bmoves(UBNUM, bd, BDINDEX) = bmoves(bx, bd, BDINDEX)
      bmoves(UBNUM, bd, DRINDEX) = dir
    end if
  next bd
end sub

' returns true if the human player's piece at the specified coordinates has at
' least 1 move, false if none.
function UserCanMove(row, col)
  canmove = 0
  if row > 1 and board(row, col) = PWHITE then
    if board(row-1, col) = 0 then canmove = 1
    if col > 1 then
      if board(row-1, col-1) = PBLACK then canmove = 1
    end if
    if col < BSIDE then
      if board(row-1, col+1) = PBLACK then canmove = 1
    end if
  end if
  UserCanMove = canmove
end function

' returns true if the computer's piece at the specified coordinates has at
' least 1 move, false if none.
function ComputerCanMove(row, col)
  canmove = 0
  if row < BSIDE and board(row, col) = PBLACK then
    if board(row+1, col) = 0 then canmove = 1
    if col > 1 then
      if board(row+1, col-1) = PWHITE then canmove = 1
    end if
    if col < BSIDE then
      if board(row+1, col+1) = PWHITE then canmove = 1
    end if
  end if
  ComputerCanMove = canmove
end function
  
' Return a list of all possible user moves from a given board position.
sub FindAllUserMoves moves(), nmoves
  local row, col
  nmoves = 0
  for row = BSIDE to 2 step -1
    for col = 1 to BSIDE
      if board(row, col) <> PWHITE then continue for
      if board(row-1, col) = 0 then
        inc nmoves
        moves(nmoves, 1) = row
        moves(nmoves, 2) = col
        moves(nmoves, 3) = USUP
        end if
      end if
      if col > 1 then
        if board(row-1, col-1) = PBLACK then
          inc nmoves
          moves(nmoves, 1) = row
          moves(nmoves, 2) = col
          moves(nmoves, 3) = USLEFT
        end if
      end if  
      if col < BSIDE then
        if board(row-1, col+1) = PBLACK then
          inc nmoves
          moves(nmoves, 1) = row
          moves(nmoves, 2) = col
          moves(nmoves, 3) = USRIGHT
        end if
      end if
    next col
  next row
end sub
        
' Select the next available user move in a cycle
sub SelectUserMove moves(), nmoves, umove
  local z$, cmd
  if nmoves = 0 then exit sub
  HiliteMove moves(), selected_move
  do
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(ucase$(z$))
    select case cmd
      case LEFT
        inc selected_move, -1
        if selected_move < 1 then selected_move = nmoves
        HiliteMove moves(), selected_move
      case RIGHT
        inc selected_move
        if selected_move > nmoves then selected_move = 1
        HiliteMove moves(), selected_move
      case UP, ENTER, CR
        umove = selected_move
        exit sub
      case ESC
        Quit
    end select
  loop
end sub

' hilite the selected User move arrow
sub HiliteMove moves(), umove
  local row, col, dir, cx, cy, rx, ry, pm, fs, ss
  local float angle
  fs = BWIDTH - 2*BMARG
  ss = fs\3
  pm = prev_selected_move
  if pm > 0 then
    row = moves(pm, 1) : col = moves(pm, 2) : dir = moves(pm, 3)
    cx = bx + (col-1)*CSIZE + CSIZE\2
    cy = by + (row-1)*CSIZE + CSIZE\2
    select case dir
      case USUP
        rx = cx+5
        ry = cy - ss
        angle = 90.0
      case USLEFT
        rx = cx + lraoff
        ry = cy - lraoff
        angle = 135.0
      case USRIGHT
        rx = cx + lraoff
        ry = cy - lraoff
        angle = 45.0
    end select
    DrawArrow apars1(), rx, ry, angle, pm, 2
  end if
  row = moves(umove, 1) : col = moves(umove, 2) : dir = moves(umove, 3)
  cx = bx + (col-1)*CSIZE + CSIZE\2
  cy = by + (row-1)*CSIZE + CSIZE\2
  select case dir
    case USUP
      rx = cx+5
      ry = cy - ss
      angle = 90.0
    case USLEFT
      rx = cx + lraoff
      ry = cy - lraoff
      angle = 135.0
     case USRIGHT
      rx = cx + lraoff
      ry = cy - lraoff
      angle = 45.0
  end select
  DrawArrow apars1(), rx, ry, angle, umove, -2
  prev_selected_move = umove
end sub  

' Execute the selected move with selected piece
sub DoUserMove moves(), move
  local row, col, dir
  row = moves(move, 1) : col = moves(move, 2) : dir = moves(move, 3)
  board(row, col) = 0
  select case dir
    case USUP
      board(row-1, col) = PWHITE
    case USLEFT
      board(row-1, col-1) = PWHITE
    case USRIGHT
      board(row-1, col+1) = PWHITE
  end select
  selected_row = 0 : selected_col = 0
  DrawBoard
end sub

' Given a board scenario, find the matching box and return the box's index.
function FindMatchingBox()
  local bx, first, last
  if ply = 2 then
    first = 1 : last = 2
  else if ply = 4 then
    first = 3 : last = 13
  else if ply >= 6 then
    first = 14 : last = 24
  end if
  for bx = first to last
    if BoxMatchesBoard(bx) then
      FindMatchingBox = bx
      exit function
    end if
  next bx
  FindMatchingBox = 0
end function

' Scan the piece contents of a box and compare it to the board
' This will only work after a user move, since the boxes only cover even plies.
function BoxMatchesBoard(n)
  local row, col, bcol
  for row = 1 to BSIDE 
    for col = 1 to BSIDE
      bcol = col 
      if bboards(n, row, bcol) <> board(row, col) then
        BoxMatchesBoard = 0
        exit function
      end if
    next col
  next row
  BoxMatchesBoard = 1
end function
        
' Hilite the matching box
sub HiliteBox n
  local x, y, brow, bcol
  if n = 0 then exit sub
  if prev_matching_box > 0 then
    brow = 1
    bcol = prev_matching_box
    if prev_matching_box > 12 then
      brow = 2
      bcol = n-12
    end if
    x = sx + (bcol-1)*(BWIDTH+BSEP)
    y = sy + (brow-1)*(BHEIGHT+BSEP)
    box x-2, y-2, BWIDTH+4, BHEIGHT+4, 2, rgb(black)
  end if
  brow = 1
  bcol = n
  if n > 12 then
    brow = 2
    bcol = n - 12
  end if
  x = sx + (bcol-1)*(BWIDTH+BSEP)
  y = sy + (brow-1)*(BHEIGHT+BSEP)
  box x-2, y-2, BWIDTH+4, BHEIGHT+4, 2, rgb(red)
  prev_matching_box = n
end sub

' Choose the computer move from existing choices and perform the move
' The parameter 'ux' points to the box that is shown
' magnified to the left of the board. The parameter 'bx' is the original box
' number from which the magnified box was derived.
sub DoComputerMove ux, bx
  local i, n, k, sq, row, col, dir
  local m$
  MakeUseBox bx
  DrawBox UBNUM, bx, MAGBOX_X, MAGBOX_Y, MAG, ply, 0
  do
    n = RandInt(1, NBCOLS)
    if beads(bx, n) > 0 then exit do
  loop
  inc cptr
  cmoves(cptr, 1) = bx
  cmoves(cptr, 2) = n
  m$ = "Computer chose " + mcolors$(n) + " move"
  WriteTacticalMessage m$, -1
  sq = bmoves(bx, n, SQINDEX)
  row = (sq-1)\3 + 1
  col = ((sq-1) mod 3) + 1
  dir = bmoves(bx, n, DRINDEX)
  board(row, col) = 0
  select case dir
    case ADOWN
      board(row+1, col) = PBLACK
    case ALEFT
      board(row+1, col-1) = PBLACK
    case ARIGHT
      board(row+1, col+1) = PBLACK
  end select
  DrawBoard
end sub

' See if one side or the other has won the game.
' Winning the game is done with one of two methods:
'  1. Advance a pawn to the opponent's starting row
'  2. Make it impossible for the opponent to move. (includes capturing all opponent's pieces)
' The function returns PBLACK or PWHITE if there is a winner or zero if not.
function IsWinner()
    local row, col
    for col = 1 to BSIDE
      if board(1, col) = PWHITE then
        IsWinner = PWHITE
        exit function
      end if
      if board(3, col) = PBLACK then
        IsWinner = PBLACK
        exit function
      end if
    next col
  for row = 1 to BSIDE
    for col = 1 to BSIDE
      if board(row, col) = mover then
        if mover = PWHITE then
          if UserCanMove(row, col) then
            IsWinner = 0
            exit function
          end if
        else
          if ComputerCanMove(row, col) then
            IsWinner = 0
            exit function
          end if
        end if
      end if
      next col
    next row
    IsWinner = 3 - mover
end function

' After the computer loses a game, train it by removing a bead in
' its move chain. This is usually the last bead chosen, but if the
' box for that bead is empty, remove the bead in its predecessor box.
sub TrainComputer
  local lbead, lbox, i, m$
  lbox = cmoves(cptr, 1)
  lbead = cmoves(cptr, 2)
  if beads(lbox, lbead) > 0 then
    beads(lbox, lbead) = 0
  else
    lbox = cmoves(cptr-1, 1)
    lbead = cmoves(cptr-1, 2)
    beads(lbox, lbead) = 0
  end if
  m$ = "Trainer removed " + mcolors$(lbead) + " bead from box " + str$(lbox)
  WriteMessage m$, 1
end sub

' Draw the Boxes
sub DrawBoxes mag
  local bx, brow, row, col, nbl, x, y, tw, n, ry, cx, id, p
  nbl = NUMB\2
  tw = nbl*mag*BWIDTH + (nbl-1)*mag*BSEP
  sx = mm.hres\2 - tw\2
  sy = 100
  n = 0
  for brow = 1 to 2
    y = sy + (brow-1)*(mag*BHEIGHT+mag*BSEP)
    for bx = 1 to nbl
      inc n
      p = 2
      if n > NB_TWO then p = 4
      if n > (NB_TWO+NB_FOUR) then p = 6
      x = sx + (bx-1)*(mag*BWIDTH+mag*BSEP)
      DrawBox n, bx, x, y, mag, p, 0
    next bx
  next brow
end sub

' Draw a Single Box
'  n is the box number; x,y are the coordinates of the tlc of the location
'  rn is the 'true' box number (not the virtual one)
'  mag is the magnification factor; p is the ply number to display
'  if r is true, erase the box instead of drawing it.
sub DrawBox n, rn, x, y, mag, p, r
  local row, col, dn
  local bboard(BSIDE*BSIDE)
  if r = 1 then
    box x-2, y-2, mag*(BWIDTH+4), mag*(BHEIGHT+4), 2, rgb(black), rgb(black)
    exit sub
  end if
  box x-2, y-2, BWIDTH+4, BHEIGHT+4, 2, rgb(black)
  box x, y, mag*BWIDTH, mag*BHEIGHT,, rgb(white), rgb(white)
  fs = mag*BWIDTH - 2*mag*BMARG
  ss = fs\3
  box x+mag*BMARG, y+mag*BMARG, fs, fs,, rgb(blue)
  for row = 1 to BSIDE
    ry = y+mag*BMARG + (row-1)*ss
    line x+mag*BMARG, ry, x+fs+2*mag, ry,, rgb(blue)
  next row
  for col = 1 to BSIDE
    rx = x+mag*BMARG + (col-1)*ss
    line rx, y+mag*BMARG, rx, y+fs+2*mag,, rgb(blue)
  next col
  text x+mag*BWIDTH\2, y+mag*BHEIGHT+2, str$(p), "CB", 5, mag, rgb(blue), -1
  dn = n
  if n = UBNUM then dn = rn
  text x+mag*BWIDTH-1, y+mag*BHEIGHT-1, str$(dn), "RB", 7,, rgb(black), -1
  DrawPieces n, x+mag*BMARG, y+mag*BMARG, ss, mag
  DrawMoves n, x+mag*BMARG, y+mag*BMARG, ss, mag
  DrawBeads n, x+mag*BMARG, ry, mag
end sub

' Draw the Pieces in a box
sub DrawPieces n, x, y, ss, mag
  local row, col, cx, cy, r
  r = int(0.35*ss)
  lraoff = int(r*cos(45))
  for row = 1 to BSIDE
    cy = y + (row-1)*ss + ss\2
    for col = 1 to BSIDE
      cx = x + (col-1)*ss + ss\2
      select case bboards(n, row, col)
        case PBLACK : circle cx, cy, r,,, rgb(black), rgb(blue)
        case PWHITE : circle cx, cy, r,,, rgb(black), rgb(yellow)
      end select
    next col
  next row
end sub

' Draw the Moves in a box
sub DrawMoves n, x, y, ss, mag
  local row, col, cx, cy
  local bd, sq, bc, dir, rx, ry
  local float ang
  for bd = 1 to NBCOLS
    if beads(n, bd) > 0 then
      bc = bmoves(n, bd, BDINDEX)
      if bc > 0 then
        sq = bmoves(n, bd, SQINDEX)
        row = (sq-1)\BSIDE + 1
        col = ((sq-1) mod BSIDE) + 1
        cy = y + (row-1)*ss + ss\2
        cx = x + (col-1)*ss + ss\2
        dir = bmoves(n, bd, DRINDEX)
        select case dir
          case ALEFT
            ang = 225.0
            rx = cx - lraoff
            ry = cy + lraoff
          case ARIGHT
            ang = 315.0
            rx = cx + lraoff
            ry = cy + lraoff
          case ADOWN
            ang = 270.0
            rx = cx
            ry = cy + ss\2
        end select
        DrawArrow apars1(), rx, ry, ang, bd, mag
      end if
    end if
  next bd
end sub

' Draw the specified arrow
sub DrawArrow apars(), x, y, angle as float, bd, umag
  local ex, ey, tl, hx, hy, ec, fc, mag
  local bx1, by1, bx2, by2, sx1, sy1, sx2, sy2
  local ax1, ay1, ax2, ay2, tx, ty
  local float pangle, hshth, hhdth
  local xv(ANVERT), yv(ANVERT)
  mag = abs(umag)
  if bd = 0 then
    ec = rgb(white)
    fc = rgb(white)
  else
    ec = rgb(black)
    fc = bcolors(bd)
    if umag < 0 then
      ec = rgb(red)
      fc = rgb(red)
    end if
  end if
  tl = mag*(apars(2)+apars(3))
  ex = int(x + tl*cos(angle) + 0.5)        ' tip of arrow
  ey = int(y - tl*sin(angle) + 0.5)
  hx = int(x + mag*apars(2)*cos(angle) + 0.5)    ' end of shaft
  hy = int(y - mag*apars(2)*sin(angle) + 0.5)
  pangle = angle + 90.0
  hshth = 0.5*mag*apars(1)
  bx1 = int(x + hshth*cos(pangle) + 0.5)    ' base vertices
  by1 = int(y - hshth*sin(pangle) + 0.5)
  bx2 = int(x - hshth*cos(pangle) + 0.5)
  by2 = int(y + hshth*sin(pangle) + 0.5)
  sx1 = int(hx + hshth*cos(pangle) + 0.5)    ' end of shaft vertices
  sy1 = int(hy - hshth*sin(pangle) + 0.5)
  sx2 = int(hx - hshth*cos(pangle) + 0.5)
  sy2 = int(hy + hshth*sin(pangle) + 0.5)
  ax1 = int(hx + mag*apars(4)*cos(pangle) + 0.5)    ' base of head vertices
  ay1 = int(hy - mag*apars(4)*sin(pangle) + 0.5)
  ax2 = int(hx - mag*apars(4)*cos(pangle) + 0.5) 
  ay2 = int(hy + mag*apars(4)*sin(pangle) + 0.5)
  xv(1) = bx1 : yv(1) = by1
  xv(2) = bx2 : yv(2) = by2
  xv(3) = sx2 : yv(3) = sy2
  xv(4) = ax2 : yv(4) = ay2
  xv(5) = ex  : yv(5) = ey
  xv(6) = ax1 : yv(6) = ay1
  xv(7) = sx1 : yv(7) = sy1
  xv(8) = bx1 : yv(8) = by1
  polygon 8, xv(), yv(), ec, fc
end sub

' Draw the beads that are currently in box 'n'
sub DrawBeads n, x, y, mag
  local h = mag*(BHEIGHT-BWIDTH)\4
  local vs = h\(NBCOLS+1)
  local i, bx, by, c
  bx = x + mag*BEAD_XOFF
  by = y + mag*BEAD_YOFF + h
  for i = 1 to NBCOLS
    if beads(n, i) > 0 then
      inc by, h
      c = bcolors(i)
      circle  bx, by, mag*BEAD_RAD,,, rgb(black), c
    end if
  next i
end sub

' Return a uniformly-distributed random integer in the closed range a to b.
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

' Quit the program and close the debug file
sub Quit
'  close #1
  cls
  end
end sub

' Write a message and pause for 'p' seconds, then erase it. If
' 'p' is negative, then don't erase the message.
sub WriteMessage m$, p
  text mm.hres\2, MSG_Y, space$(60), "CT", 4
  text mm.hres\2, MSG_Y, m$, "CT", 4,, rgb(red)
  if p < 0 then exit sub
  pause 1000*p
  text mm.hres\2, MSG_Y, space$(60), "CT", 4
end sub

' Write a tactical message about the computer's move
' (there is only room for about 30 chars)
sub WriteTacticalMessage m$, p
  text 40, BYPOS-2, space$(30), "LB", 7
  text 40, BYPOS-2, m$, "LB", 7
  if p < 0 then exit sub
  pause 1000*p
  text 40, BYPOS-2, space$(30), "LB", 7
end sub
  
' Show a summary of games played, winners and losers, and
' a graph.
sub DrawGamesSummary
  local x, y, m$, i
  local gwline$ = ""
  box GSX, BYPOS, GSW, GSH,, rgb(black), rgb(black)
  box GSX, BYPOS, GSW, GSH
  gs = MCHARS*2
  y = BYPOS+2
  text GSX+GSW\2, y, "Summary of Games", "CT",,, rgb(green)
  x = GSX+5
  inc y, 20
  text x, y, "'U' = Human"
  inc y, 15
  text x, y, "'C' = Computer"
  inc y, 20
  m$ = "Games Played: " + str$(ngames)
  text x, y, m$
  inc y, 20
  for i = 1 to ngames
    if games(i) = PWHITE then
      gwline$ = gwline$ + "U"
    else
      gwline$ = gwline$ + "C"
    end if
    if len(gwline$) >= MCHARS then
      text x, y, gwline$
      gwline$ = ""
      inc y, 15
    end if
  next i
  if len(gwline$) > 0 then
    text x, y, gwline$
  end if
  DrawWinnerGraph
end sub

' Draw a graph of game winners. Game index on X axis, Winner on Y axis, where a computer
' win makes graph rise, and a computer loss makes graph fall.
sub DrawWinnerGraph
  local x, y, i, px, py, ccw
  local float xscale, yscale, ystart
  xscale = (1.0*WGW)/(1.0*MAX_GAMES)
  yscale = (1.0*WGH)/(2.0*MCHARS)
  ystart = 0.5*WGH
  box WGX, WGY, WGW, WGH
  text WGX+WGW\2, WGY+WGH+1, "Games", "CT"
  text WGX-WGM\2, WGY+WGH\2, "CWins", "CMV"
  px = WGX
  py = WGY+WGH - int(ystart)
  ccw = 0
  for i = 1 to ngames
    if games(i) = PWHITE then inc ccw, -1
    if games(i) = PBLACK then inc ccw
    x = WGX + int(xscale*i + 0.5)
    y = WGY+WGH - int(ystart) - int(yscale*ccw + 0.5)
    line px, py, x, y,, rgb(blue)
    px = x : py = y
  next i
end sub

' return the number of beads in a box
function GetNumBeads(bx)
  local n, i
  n = 0
  for i = 1 to NBCOLS
    if beads(bx, i) > 0 then inc n
  next i
  GetNumBeads = n
end function

' Show the help messages (long)
sub ShowHelp
  local z$
  cls
  text mm.hres/2, 1, "How to Use the Hexapawn Program", "CT", 4,, rgb(green)
  print @(0, 20)
  print "'Hexapawn' is a trivial game invented by Martin Gardner and described in the March, 1962"
  print "issue of Scientific American magazine. The purpose of this simple game was to demonstrate"
  print "machine learning using a 'computer' made of matchboxes and beads. The possibility of doing"
  print "such a thing had been earlier proved by Donald Michie with his Tic-Tac-Toe learning machine"
  print "that used more than 300 matchboxes. Hexapawn is such a simple game that it only requires"
  print "24 matchboxes to learn how to play a perfect game, starting from completely random play."
  print ""
  print "This computer program lets you watch the 'matchbox' learning machine gradually get better"
  print "until it will beat you every time at Hexapawn. The matchboxes are emulated using graphics,"
  print "of course.
  print ""
  print "The game of Hexapawn is played on a 3x3 chessboard. You play the white pawns, which start"
  print "at the bottom of the board, and the computer plays the black pawns, which start at the top."
  print "Similar to real chess, pawns can only move forward or capture diagonally, but never move"
  print "backwards. Unlike chess, pawns can only move forward one square on their first move, not two."
  print "The game is won when one side or the other advances a pawn to their opponent's home row,"
  print "or when it is the opponent's turn and all their pawns are blocked from moving, or when all"
  print "the opponent's pawns have been captured. No game can last more than 4 moves for each side,"
  print "otherwise known as 8 plies."
  print ""
  print "For a more pleasing color scheme, this program draws the 'white' pawns as yellow, and the"
  print "'black' pawns as blue. This image shows the starting position for a game of Hexapawn."
  print ""
  print "You will use the keyboard arrow keys to move your pawns. When it is your turn to move, a"
  print "set of colored arrows will be shown that correspond to all your possible moves. One of the"
  print "arrows will be colored bright red. This is your currently selected move. To change to a"
  print "different move, press, the LEFT, RIGHT, or DOWN arrow key; then press UP to make your move."

  InitGame
  DrawBoard
  text mm.hres\2, mm.vres-2, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""

  cls
  print "The 24 matchboxes are shown at the top of the screen. Each box represents a position in the"
  print "game. The user always goes first, so the computer's moves are always on EVEN plies: 2, 4, or"
  print "6. The ply number is shown in a large font at the bottom of each 'matchbox'. You will note"
  print "That there are only 2 matchboxes for ply 2, even there are 3 possible plays for the user's"
  print "first move. This is because if the user plays the pawn at the right, the entire board for"
  print "this and all subsequent moves in a game are just the left/right mirror images of those that"
  print "are shown on the 24 matchboxes. So if you move the rightmost pawn on your first move, the"
  print "computer will flip all the 24 boxes."
  InitBeads
  DrawBoxes 1
  print @(0, 285)
  print "The colored arrows in each box show the computer's possible moves from that board position."
  print "Each move corresponds to a colored bead inside the box. Those beads are shown at the lower"
  print "left of each box. At the start of training, the computer makes random choices among legal"
  print "moves."
  print ""
  print "The training mechanism which allows the computer to improve its play through multiple games"
  print "consists of 'punishment' when it loses a game. After a loss to the human, the training"
  print "module, which has been keeping track of moves during the game, removes the last bead that"
  print "led to a losing move. After multiple games, ALL of the beads that produce losing play will"
  print "have been removed, and only winning moves remain. As beads are removed, they disappear from"
  print "the boxes, and so do their corresponding moves."
  print ""
  print "During play, when the computer moves, you will see a magnified image of the box it is using"
  print "appear to the left of the board. The box in the box array is also hilited with a red 
  print "outline."
  print ""
  print "It generally takes about 25-30 games before the computer plays a perfect game. The game"
  print "summary display at lower right shows you the sequence of game winners and a graph showing"
  print "the progress toward perfection. You will also see beads vanish along with their move"
  print "arrows.
  print ""
  print "To make the computer start its learning from scratch again, stop and restart the program."

  text mm.hres\2, mm.vres-2, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' ===================================
' Only debug methods past this point
' ===================================

sub PrintNumBeads bx
  local n, i
  n = 0
  for i = 1 to NBCOL
    if beads(bx, i) > 0 then inc n
  next i
end sub

' ==========================
' Only data past this point
' ==========================

' Box contents. The first data statement per box is for the piece positions.
' The second data statement per box is for the moves.
' Box order is raster: horizontally and then next row.

'------------------------------------------------------------------------------
' These are the box labels and contents as shown in Martin Gardener's article
' The ones shown in the VSauce2 YouTube video are somewhat different, and in
' fact, incorrect!
'------------------------------------------------------------------------------

' Ply 2 boxes
data PBLACK, PBLACK, PBLACK, PWHITE, 0, 0, 0, PWHITE, PWHITE
data 2, ADOWN, 2, ALEFT, 3, ADOWN, 0, 0

data PBLACK, PBLACK, PBLACK, 0, PWHITE, 0, PWHITE, 0, PWHITE
data 1, ARIGHT, 1, ADOWN, 0, 0, 0, 0

' Ply 4 boxes
data PBLACK, 0, PBLACK, PBLACK, PWHITE, 0, 0, 0, PWHITE
data 1, ARIGHT, 3, ALEFT, 3, ADOWN, 4, ADOWN

data 0, PBLACK, PBLACK, PWHITE, PBLACK, 0, 0, 0, PWHITE
data 2, ALEFT, 3, ADOWN, 5, ADOWN, 0, 0

data PBLACK, 0, PBLACK, PWHITE, PWHITE, 0, 0, PWHITE, 0
data 1, ARIGHT, 3, ALEFT, 3, ADOWN, 0, 0

data PBLACK, PBLACK, 0, PWHITE, 0, PWHITE, 0, 0, PWHITE
data 2, ALEFT, 2, ADOWN, 2, ARIGHT, 0, 0

data 0, PBLACK, PBLACK, 0, PBLACK, PWHITE, PWHITE, 0, 0
data 2, ARIGHT, 5, ALEFT, 5, ADOWN, 0, 0

data 0, PBLACK, PBLACK, PBLACK, PWHITE, PWHITE, PWHITE, 0, 0
data 2, ARIGHT, 3, ALEFT, 0, 0, 0, 0

data PBLACK, 0, PBLACK, PBLACK, 0, PWHITE, 0, PWHITE, 0
data 4, ADOWN, 4, ARIGHT, 0, 0, 0, 0

data PBLACK, PBLACK, 0, PWHITE, PWHITE, PBLACK, 0, 0, PWHITE
data 1, ARIGHT, 2, ALEFT, 0, 0, 0, 0

data 0, PBLACK, PBLACK, 0, PWHITE, 0, 0, 0, PWHITE
data 3, ALEFT, 3, ADOWN, 0, 0, 0, 0

data 0, PBLACK, PBLACK, 0, PWHITE, 0, PWHITE, 0, 0
data 3, ALEFT, 3, ADOWN, 0, 0, 0, 0

data PBLACK, 0, PBLACK, PWHITE, 0, 0, 0, 0, PWHITE
data 3, ADOWN, 0, 0, 0, 0, 0, 0

' Ply 6 boxes

data 0, 0, PBLACK, PBLACK, PBLACK, PWHITE, 0, 0, 0
data 4, ADOWN, 5, ADOWN, 0, 0, 0, 0

data PBLACK, 0, 0, PWHITE, PWHITE, PWHITE, 0, 0, 0
data 1, ARIGHT, 0, 0, 0, 0, 0, 0

data 0, PBLACK, 0, PBLACK, PWHITE, PWHITE, 0, 0, 0
data 2, ARIGHT, 4, ADOWN, 0, 0, 0, 0

data 0, PBLACK, 0, PWHITE, PWHITE, PBLACK, 0, 0, 0
data 2, ALEFT, 6, ADOWN, 0, 0, 0, 0

data PBLACK, 0, 0, PBLACK, PBLACK, PWHITE, 0, 0, 0
data 4, ADOWN, 5, ADOWN, 0, 0, 0, 0

data 0, 0, PBLACK, PWHITE, PBLACK, PBLACK, 0, 0, 0
data 5, ADOWN, 6, ADOWN, 0, 0, 0, 0

data 0, 0, PBLACK, PBLACK, PWHITE, 0, 0, 0, 0
data 3, ALEFT, 3, ADOWN, 4, ADOWN, 0, 0

data 0, PBLACK, 0, PWHITE, PBLACK, 0, 0, 0, 0
data 2, ALEFT, 5, ADOWN, 0, 0, 0, 0

data 0, PBLACK, 0, 0, PBLACK, PWHITE, 0, 0, 0
data 2, ARIGHT, 5, ADOWN, 0, 0, 0, 0

data PBLACK, 0, 0, PBLACK, PWHITE, 0, 0, 0, 0
data 1, ARIGHT, 4, ADOWN, 0, 0, 0, 0

data 0, 0, PBLACK, 0, PWHITE, PBLACK, 0, 0, 0
data 3, ALEFT, 6, ADOWN, 0, 0, 0, 0

